/*-*- comment-column: 40 -*-*/
/* ipgen -- IP test packet description generator, output module. 
 *
 * Author: Juergen Nickelsen <nickelsen@condat.de>
 * Copyright (C) Condat AG 2000
 * $Id: output.c,v 1.5 2000/07/13 20:25:25 ni Exp $
 */


#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipgen.h"


/** Compute Internet checksum over two areas. The first of the two
 * areas must have even length.
 * @param p1      pointer to area1
 * @param count1  size of area1
 * @param p2      pointer to area2
 * @param count2  size of area2
 * @return        checksum
 */
unsigned short inet_checksum2(void *p1, int count1, void *p2, int count2)
{
    long sum = 0 ;
    unsigned short *addr ;

    /* Sum first area. */
    addr = (unsigned short *) p1 ;
    while (count1 > 1) {
	sum += *addr++ ;
	count1 -= 2 ;
    }

    /* Sum second area. */
    addr = (unsigned short *) p2 ;
    while (count2 > 1) {
	sum += *addr++ ;
	count2 -= 2 ;
    }

    if (count2 > 0) {
	sum += * (unsigned char *) addr ;
    }

    while (sum >> 16) {
	sum = (sum & 0xffff) + (sum >> 16) ;
    }

    return ~sum ;
}


int bytes_printed = 0 ;			/* Number of bytes printed
					 * on output in the current
					 * line. */

/** Print a byte to the output. Insert newlines and tabs as
 * necessary. Obey output_width.
 * @param byte         data byte to print.
 * @param print_comma  print comma after value
 */
void print_byte(unsigned char byte, int print_comma)
{
    if (bytes_printed == 0) {
	printf("%s", OUTPUT_INDENT) ;
    }
    printf("0x%02x%s", byte, print_comma ? "," : "") ;
    if (++bytes_printed == output_width) {
	putchar('\n') ;
	bytes_printed = 0 ;
    } else {
	putchar(' ') ;
    }
    fflush(stdout) ;
}


/** Ensure that there is a newline after printing output bytes. If n
 * is non-zero, print as many additional newlines, i. e. n is the
 * number of empty lines.
 * @param n   number of empty lines
 */
void have_newline(int n)
{
    if (bytes_printed != 0) {
	printf("\n") ;
	bytes_printed = 0 ;
    }
    while (n--) {
	printf("\n") ;
    }
}



/** Print an assembled packet.
 * @param pkt_ptr  pointer to packet struct
 */
void output_packet(struct IP_PKT *pkt)
{
    struct sduhdr {			/* Header of primitive SDU. */
	/* Values in little endian! */
	unsigned char bitlen_lo ;	/* Low byte of length in bits. */
	unsigned char bitlen_hi ;	/* High byte of length in bits. */
	unsigned char offset_lo ;	/* Low byte of offset in bits. */
	unsigned char offset_hi ;	/* High byte of offset in bits. */
    } sdu ;
    struct iphdr {			/* IP header with options (packed). */
	u_char	ip_vhl;			/* version and header length */
	u_char	ip_tos;			/* type of service */
	u_short	ip_len;			/* total length */
	u_short	ip_id;			/* identification */
#define	IP_OFFMASK 0x1fff		/* mask for fragmenting bits */
	u_short	ip_off;			/* fragment offset field */
	u_char	ip_ttl;			/* time to live */
	u_char	ip_p;			/* protocol */
	u_short	ip_sum;			/* checksum */
	struct	in_addr ip_src,ip_dst;	/* source and dest address */
	char options[40] ;
    } ip ;
    struct udp {			/* UDP with IP pseudo header. */
	/* IP pseudo header. */
	struct in_addr ip_src ;
	struct in_addr ip_dst ;
	u_char ip_zero ;
	u_char ip_p ;
	u_short ip_ulen ;
	/* UDP header. */
	u_short	uh_sport;		/* source port */
	u_short	uh_dport;		/* destination port */
	u_short	uh_ulen;		/* udp length */
	u_short	uh_sum;			/* udp checksum */
    } udp ;
    struct icmp {
	u_char  type ;			/* ICMP type. */
	u_char  code ;			/* ICMP code. */
	u_short sum ;			/* ICMP checksum. */
	u_long  id_seq ;		/* Echo ID and sequence or
					 * unused in the types we
					 * handle. */
    } icmp ;
    int i ;				/* Loop counter. */
    int printed = 0 ;			/* Number of characters
					 * already printed. */
    int ip_hlength = 0 ;		/* Actual number of bytes in
					 * IP header (as opposed to
					 * the IHL value). */
    int udp_length = 0 ;		/* Number of bytes in UDP header. */
    int icmp_length = 0 ;		/* Number of bytes in ICMP header. */
    int total_length ;			/* Total number of bytes to
					 * print; depends on
					 * pkt->real_length. */
    char *asrc, *adst ;			/* ASCII representation of
					 * source and destination
					 * address. */

    /* IP header with optional options.
     */

    memset(&ip, '\0', sizeof(ip)) ;
    ip.ip_vhl = (pkt->ip_v << 4) | (pkt->ip_hl & 0xf) ;
    ip.ip_tos = pkt->ip_tos ;
    if (!pkt->has_ip_len) {
	int transport_len ;		/* Length of transport header. */

	if (pkt->ip_p == IPPROTO_ICMP || pkt->ip_p == IPPROTO_UDP) {
	    transport_len = 8 ;
	} else {
	    /* Otherwise we have no transport header (or perhaps in
	     * the data). */
	    transport_len = 0 ;
	}
	pkt->ip_len = pkt->ip_hl * 4 + transport_len + pkt->data_len ;
    }
    ip.ip_len = htons(pkt->ip_len) ;
    ip.ip_id  = htons(pkt->ip_id) ;
    ip.ip_off = htons(  (pkt->ip_off & IP_OFFMASK)
		      | (pkt->ip_flags << 13)) ;
    ip.ip_ttl = pkt->ip_ttl ;
    ip.ip_p   = pkt->ip_p ;
    ip.ip_src = pkt->ip_src ;
    ip.ip_dst = pkt->ip_dst ;

    ip_hlength = MAX((ip.ip_vhl & 0xf) * 4, MIN_IP_HEADER_LENGTH) ;

    /* Fill option space with random bytes. */
    for (i = 0; i < ((int) pkt->ip_hl - 5) * 4; i++) {
	ip.options[i] = random() ;
    }

    /* If a checksum has been specified, use that one. Otherwise
     * calculate. */
    if (pkt->has_ip_sum) {
	ip.ip_sum = htons(pkt->ip_sum) ;
    } else {
	ip.ip_sum = 0 ;
	/* Evidently we must not turn this to network order via
	 * htons(). (This gives exactly the wrong result.) This is
	 * probably because the values the checksum is calculated
	 * over are already in network order. */
	ip.ip_sum = inet_checksum2(&ip, pkt->ip_hl * 4, 0, 0) ;
    }

    /* UDP header or ICMP header, if ip_p has the corresponding
     * value.
     */

    if (ip.ip_p == IPPROTO_UDP) {
	/* Fill and print UDP header including IP pseudo header. */
	udp.uh_sport = htons(pkt->udp_sport) ;
	udp.uh_dport = htons(pkt->udp_dport) ;
	if (pkt->has_udp_ulen) {
	    udp.uh_ulen = htons(pkt->udp_ulen) ;
	} else {
	    udp.uh_ulen = htons(8 + pkt->data_len) ;
	}
	if (pkt->has_udp_sum) {
	    udp.uh_sum = htons(pkt->udp_sum) ;
	} else {
	    udp.ip_src  = ip.ip_src ;
	    udp.ip_dst  = ip.ip_dst ;
	    udp.ip_zero = 0 ;
	    udp.ip_p    = ip.ip_p ;
	    udp.ip_ulen = udp.uh_ulen ;
	    udp.uh_sum  = 0 ;
	    udp.uh_sum  = inet_checksum2(&udp, sizeof(udp),
					 pkt->data, pkt->data_len) ;
	}
	udp_length = 8 ;
    } else if (ip.ip_p == IPPROTO_ICMP) {
	icmp.type   = pkt->icmp_type ;
	icmp.code   = pkt->icmp_code ;
	if (pkt->has_icmp_id) {
	    icmp.id_seq = htonl(  (pkt->icmp_id << 16)
				| (pkt->icmp_seq & 0xffff)) ;
	} else {
	    icmp.id_seq = htonl(pkt->icmp_unused) ;
	}
	if (pkt->has_icmp_sum) {
	    icmp.sum = htons(pkt->icmp_sum) ;
	} else {
	    icmp.sum = 0 ;
	    icmp.sum = inet_checksum2(&icmp, sizeof(icmp),
				      pkt->data, pkt->data_len) ;
	}
	icmp_length = 8 ;
    }


    /* Finally print the packet description. */

    /* If a `real_length' has been specified, use this value. */
    if (pkt->real_length >= 0) {
	total_length = pkt->real_length ;
    } else {
	total_length = pkt->ip_len ;
    }

    /* SDU header */
    sdu.bitlen_hi = (total_length * 8) >> 8 ;
    sdu.bitlen_lo = (total_length * 8) & 0xff ;
    sdu.offset_hi = 0 ;
    sdu.offset_lo = 0 ;

    have_newline(1) ;
/*     printf("DECLARATION(%s\n", pkt->name) ; */
/*     printf("BEGINARRAY(%s, %d)\n", pkt->name, total_length + sizeof(sdu)) ; */
    printf("FIELD(%s)\n", pkt->name) ;
    if (pkt->comment) {
	printf("/* %s */\n", pkt->comment) ;
    }

    /* SDU header */
    for (i = 0; i < sizeof(sdu); i++) {
	print_byte(*((unsigned char *) &sdu + i), 1) ;
    }
    have_newline(0) ;
    
    /* IP header */
    asrc = ck_strdup(inet_ntoa(pkt->ip_src)) ;
    adst = ck_strdup(inet_ntoa(pkt->ip_dst)) ;
    printf("%s/* IP proto %d len %d %s -> %s */\n",
	   OUTPUT_INDENT, pkt->ip_p, ntohs(ip.ip_len),
	   asrc, adst) ;
    free(asrc) ;
    free(adst) ;
    for (i = 0; i < ip_hlength; i++) {
	if (printed++ >= total_length) {
	    break ;
	}
	print_byte(*((unsigned char *) &ip + i), printed < total_length) ;
    }
    have_newline(0) ;

    if (ip.ip_p == IPPROTO_UDP) {
	printf("%s/* UDP sport %d dport %d ulen %d */\n",
	       OUTPUT_INDENT, ntohs(udp.uh_sport), ntohs(udp.uh_dport),
	       ntohs(udp.uh_ulen)) ;
	for (i = 0; i < 8; i++) {
	    if (printed++ >= total_length) {
		break ;
	    }
	    print_byte(*((unsigned char *) &udp.uh_sport + i),
		       printed < total_length) ;
	}
    } else if (ip.ip_p == IPPROTO_ICMP) {
	printf("%s/* ICMP type %d code %d */\n",
	       OUTPUT_INDENT, icmp.type, icmp.code) ;
	for (i = 0; i < 8; i++) {
	    if (printed++ >= total_length) {
		break ;
	    }
	    print_byte(*((unsigned char *) &icmp + i),
		       printed < total_length) ;
	}
    }
    have_newline(0) ;

    if (printed < total_length) {
	printf("%s/* data */\n", OUTPUT_INDENT) ;
	for (i = 0 ; printed++ < total_length; i++) {
	    if (pkt->data_len) {
		print_byte(pkt->data[i % pkt->data_len],
			   printed < total_length) ;
	    } else {
		print_byte(random(), printed < total_length) ;
	    }
	}
    }
    have_newline(0) ;

/*     printf("ENDARRAY\n") ; */
    printf("ENDFIELD(%s, %d)\n", pkt->name, total_length + sizeof(sdu)) ;
}


/* EOF */
